unit Grids;

interface

uses SysUtils, Engine, DePack, Forms, Controls, StdCtrls, Windows, Classes, ProgBar,
     Math, Libraries;

type
 TBlockIndex = record
  Lt, Brk: Byte;
 end;

 TGridObj = record
  Flags: Byte;
  Blocks: array of TBlockIndex;
 end;

 TCell = record
  Offset: WORD;
  Repeated: Boolean;
  ObjCount: Byte;
  Objects: array of TGridObj;
 end;

 TGrid = record
  Lba2: Boolean;
  LibIndex: Byte;
  FragIndex: Byte;
  LibUsage: array[0..31] of Byte;
  Cells: array of TCell;
 end;

 TFragment = record
  X, Y, Z: Byte;
  LibIndex: Byte;
  Blocks: array of array of array of TBlockIndex;
 end;

 TFrameOpts = set of (foNormal, foSelect);

 TMapItem = record
  Idx: TBlockIndex;
  BrkNr: Integer;
  Shape: Byte;
  Frame: TFrameOpts;
  FrLines: TFrameLines;
 end;

 TGridMap = array of array of array of TMapItem;

 TPiece = record
  X, Y, Z: Integer;
  Map: array of array of array of TMapItem;
 end;

Function LoadGridFromStream(data: TStream; Lba2: Boolean): TGrid;
Function LoadFragFromStream(data: TStream): TFragment;
function GridToString(Grid: TGrid; Lba2: Boolean): String;
procedure SaveGrid(path: String; Index: Integer = -1);
procedure ClearMap;
Procedure FindTransparentBrick;
Procedure FindLba2Invisible(Lib: TLibrary);
procedure DataToMap;
procedure SetAllFrameLines;
Function MapToGrid: TGrid;
Function MapToFragment: TFragment;
Function LayoutToPiece(LtNr: Integer): TPiece;
Function CopyPiece(x, y, z, dx, dy, dz: Word): TPiece;
Procedure PutPiece(x, y, z: Word; Fr: TPiece; Transparent: Boolean = True);
Procedure DelPiece(x, y, z, dx, dy, dz: Integer); overload;
Procedure DelPiece(Range: TBox); overload;

implementation

uses Open, BEngine, Main, Hints, Sett, CompDialog, OpenSim, Clipping, Bricks;

Function LoadGridFromStream(data: TStream; Lba2: Boolean): TGrid;
var a, b, c, BlockCount: Integer;
    d: Byte;
begin
 with data do begin
  SetLength(Result.Cells,64*64);
  If Lba2 then begin
   Result.Lba2:= True;
   Seek(0,soBeginning);
   Read(Result.LibIndex,1);
   Read(Result.FragIndex,1);
   d:= 34;
  end
  else d:=0;
  Seek(d,soBeginning);
  for a:=0 to 64*64-1 do
   Read(Result.Cells[a].Offset,2);
  for a:=0 to 64*64-1 do begin
   Seek(Result.Cells[a].Offset+d,soBeginning);
   Read(Result.Cells[a].ObjCount,1);
   SetLength(Result.Cells[a].Objects,Result.Cells[a].ObjCount);
   for b:=0 to High(Result.Cells[a].Objects) do begin
    Read(Result.Cells[a].Objects[b].Flags,1);
    BlockCount:=(Result.Cells[a].Objects[b].Flags and $1F)+1;
    case (Result.Cells[a].Objects[b].Flags and $E0) of
     $00: SetLength(Result.Cells[a].Objects[b].Blocks,0);
     $40: begin
           SetLength(Result.Cells[a].Objects[b].Blocks,BlockCount);
           for c:=0 to BlockCount-1 do begin
            Read(Result.Cells[a].Objects[b].Blocks[c].Lt,1);
            Read(Result.Cells[a].Objects[b].Blocks[c].Brk,1);
           end;
          end;
     $80: begin
           SetLength(Result.Cells[a].Objects[b].Blocks,1);
           Read(Result.Cells[a].Objects[b].Blocks[0].Lt,1);
           Read(Result.Cells[a].Objects[b].Blocks[0].Brk,1);
          end;
    end;
   end;
  end;
 end;
end;

Function LoadFragFromStream(data: TStream): TFragment;
var a, b, c: Integer;
begin
 with data do begin
  Seek(0,soBeginning);
  Read(Result.X,1);
  Read(Result.Y,1);
  Read(Result.Z,1);
  SetLength(Result.Blocks,Result.X,Result.Y,Result.Z);
  for c:=0 to Result.Z-1 do
   for a:=0 to Result.X-1 do
    for b:=0 to Result.Y-1 do begin
     Read(Result.Blocks[a,b,c].Lt,1);
     Read(Result.Blocks[a,b,c].Brk,1);
    end;
 end; 
end;


function GridToString(Grid: TGrid; Lba2: Boolean): String;
var a, b, c: Integer;
    s: ShortString;
begin
 Result:='';
 s:='';
 for a:= 0 to 31 do s:= s + Char(Grid.LibUsage[a]);
 for a:= 0 to 64*64-1 do
  Result:= Result + GetStrWord(Grid.Cells[a].Offset);
 for a:= 0 to 64*64-1 do begin
  If Grid.Cells[a].Repeated then Continue;
  Result:= Result + Char(Grid.Cells[a].ObjCount);
  for b:= 0 to High(Grid.Cells[a].Objects) do begin
   Result:= Result + Char(Grid.Cells[a].Objects[b].Flags);
   for c:= 0 to High(Grid.Cells[a].Objects[b].Blocks) do begin
    Result:= Result + Char(Grid.Cells[a].Objects[b].Blocks[c].Lt);
    Result:= Result + Char(Grid.Cells[a].Objects[b].Blocks[c].Brk);
   end;
  end;
 end;
 If Lba2 then Result:= Char(Grid.LibIndex) + Char(Grid.FragIndex) + s + Result
 else Result:= Result + s;
end;

Function FragToString(Frag: TFragment): String;
var a, b, c: Integer;
begin
 Result:= Char(Frag.X) + Char(Frag.Y) + Char(Frag.Z);
 for c:=0 to Frag.Z-1 do
  for a:=0 to Frag.X-1 do
   for b:=0 to Frag.Y-1 do
    Result:= Result + Char(Frag.Blocks[a,b,c].Lt) + Char(Frag.Blocks[a,b,c].Brk);
end;

Procedure SaveStringToFile(st: String; path: String);
var f: File;
begin
 AssignFile(f,path);
 Rewrite(f,1);
 BlockWrite(f,st[1],Length(st));
 CloseFile(f);
end;

procedure SaveGrid(path: String; Index: Integer = -1);
var ext: String;
    VPack: TPackEntries;
    NewComp: Word;
begin
 Screen.Cursor:= crHourGlass;
 ext:= LowerCase(ExtractFileExt(path));
 If (ext = '.gr1') or (ext = '.gr2') then begin
  Grid:= MapToGrid;
  SaveStringToFile(GridToString(Grid,ext='.gr2'),path);
 end
 else if ext = '.grf' then begin
  Frag:= MapToFragment;
  SaveStringToFile(FragToString(Frag),path);
 end
 else if ext='.hqr' then begin
  If GridNow then Grid:= MapToGrid
  else Frag:= MapToFragment;
  VPack:= OpenPack(path);
  If not SaveCompDialog(IsBkg(path),OriginalComp,NewComp) then begin
   Screen.Cursor:= crDefault;
   Exit;
  end;
  If GridNow then VPack[Index]:= PackEntry(GridToString(Grid,IsBkg(path)),-1,NewComp)
  else VPack[Index]:= PackEntry(FragToString(Frag),-1,NewComp);
  If IsBkg(path) then BkgHeadFix(VPack,-1,-1,-1,-1,-1);
  OriginalComp:= NewComp;
  SavePackToFile(VPack,path);
 end
 else Exit;
 Modified:= False;
 Screen.Cursor:= crDefault;
 SysUtils.Beep;
 PutMessage(28);
 LastSaveDir:= ExtractFilePath(path);
 CurrentFile:= path;
end;

procedure ClearMap;
var a, b, c: Integer;
    temp: TMapItem;
begin
 temp.Idx.Lt:= 0;
 temp.Idx.Brk:= 0;
 temp.BrkNr:= -1;
 temp.Shape:= 0;
 temp.Frame:= [];
 temp.FrLines:= [];
 for a:=0 to HighX do
  for b:=0 to HighY do
   for c:=0 to HighZ do
    Map[a,b,c]:= temp;
end;

procedure SetFrameOpts(var Item: TMapItem);
begin
 Item.Frame:= [];
 If Item.BrkNr >= 1 then Item.Frame:= [foNormal,foSelect];
 If (((GridNow and Grid.Lba2) or not GridNow)
 and (Item.BrkNr = TransparentBrick)) or (Item.BrkNr = 0) then
  Item.Frame:= [foSelect];
end;

procedure SetFrameLines(x, y, z: Integer);
var a, b, c: Integer;
    temp: TFrameLines;
    surround: array[-1..1, -1..1, -1..1] of Boolean;
begin
 If not (foSelect in Map[x,y,z].Frame) then
  Map[x,y,z].FrLines:= []
 else begin
  temp:= [];
  If SetForm.rbNewFrEdge.Checked then //by edges
   for a:=-1 to 1 do
    for b:=-1 to 1 do
     for c:=-1 to 1 do
      If (x+a<0) or (x+a>HighX) or (y+b<0) or (y+b>HighY) or (z+c<0) or (z+c>HighZ) then
       surround[a,b,c]:= False
      else
       surround[a,b,c]:= foSelect in Map[x+a,y+b,z+c].Frame
  else     //by objects
   for a:=-1 to 1 do
    for b:=-1 to 1 do
     for c:=-1 to 1 do
      If (x+a<0) or (x+a>HighX) or (y+b<0) or (y+b>HighY) or (z+c<0) or (z+c>HighZ) then
       surround[a,b,c]:= False
      else
       surround[a,b,c]:= Map[x+a,y+b,z+c].Idx.Lt = Map[x,y,z].Idx.Lt;

  If (surround[-1,0,0] and surround[0,1,0] and not surround[-1,1,0])
     or not (surround[-1,0,0] or surround[0,1,0]) then temp:= temp + [flTopLeftBack];
  If (surround[0,0,1] and surround[0,1,0] and not surround[0,1,1])
     or not (surround[0,0,1] or surround[0,1,0]) then temp:= temp + [flTopLeftFront];
  If (surround[0,0,-1] and surround[0,1,0] and not surround[0,1,-1])
     or not (surround[0,0,-1] or surround[0,1,0]) then temp:= temp + [flTopRightBack];
  If (surround[1,0,0] and surround[0,1,0] and not surround[1,1,0])
     or not (surround[1,0,0] or surround[0,1,0]) then temp:= temp + [flTopRightFront];

  If (surround[-1,0,0] and surround[0,-1,0] and not surround[-1,-1,0])
     or not (surround[-1,0,0] or surround[0,-1,0]) then temp:= temp + [flBtmLeftBack];
  If (surround[0,0,1] and surround[0,-1,0] and not surround[0,-1,1])
     or not (surround[0,0,1] or surround[0,-1,0]) then temp:= temp + [flBtmLeftFront];
  If (surround[0,0,-1] and surround[0,-1,0] and not surround[0,-1,-1])
     or not (surround[0,0,-1] or surround[0,-1,0]) then temp:= temp + [flBtmRightBack];
  If (surround[1,0,0] and surround[0,-1,0] and not surround[1,-1,0])
     or not (surround[1,0,0] or surround[0,-1,0]) then temp:= temp + [flBtmRightFront];

  If (surround[1,0,0] and surround[0,0,1] and not surround[1,0,1])
     or not (surround[1,0,0] or surround[0,0,1]) then temp:= temp + [flVertFront];
  If (surround[-1,0,0] and surround[0,0,1] and not surround[-1,0,1])
     or not (surround[-1,0,0] or surround[0,0,1]) then temp:= temp + [flVertLeft];
  If (surround[1,0,0] and surround[0,0,-1] and not surround[1,0,-1])
     or not (surround[1,0,0] or surround[0,0,-1]) then temp:= temp + [flVertRight];
  If (surround[-1,0,0] and surround[0,0,-1] and not surround[-1,0,-1])
     or not (surround[-1,0,0] or surround[0,0,-1]) then temp:= temp + [flVertBack];

  Map[x,y,z].FrLines:= temp;
 end;
end;

procedure SetAllFrameLines;
var a, b, c: Integer;
begin
 for c:=0 to HighZ do
  for b:=0 to HighY do
   for a:=0 to HighX do
    SetFrameLines(a,b,c);
end;

procedure SetBrkNr(Lib: TLibrary; var MapItem: TMapItem);
begin
 If MapItem.Idx.Lt > 0 then begin
  If (Length(Lib) < MapItem.Idx.Lt)
  or (High(Lib[MapItem.Idx.Lt].Map) < MapItem.Idx.Brk) then begin
   If not BadLibraryMessage then MessageBox(ProgBarForm.Handle,'One or more Grid objects refer to a Layout or Brick that doesn''t exist! You should check if the Library you selected is appropriate to the Grid. Anyway, the Grid will be opened, but invalid references will be removed.',ProgramName,MB_ICONWARNING+MB_OK);
   BadLibraryMessage:= True;
   MapItem.Idx.Lt:= 0;
   MapItem.Idx.Brk:= 0;
   MapItem.BrkNr:= 0;
   MapItem.Shape:= 0;
  end
  else begin
   MapItem.BrkNr:= Lib[MapItem.Idx.Lt].Map[MapItem.Idx.Brk].Index;
   MapItem.Shape:= Lib[MapItem.Idx.Lt].Map[MapItem.Idx.Brk].Shape;
  end;
 end
 else if not GridNow and (MapItem.Idx.Brk = 0) then MapItem.BrkNr:= -1
 else MapItem.BrkNr:= 0;
end;

Procedure FindTransparentBrick;
var a: Integer;
begin
 TransparentBrick:= 0;
 For a:= 0 to High(VBricks) do
  If BrickIsEmpty(a) then begin
   TransparentBrick:= a;
   Exit;
  end;
end;

Procedure FindLba2Invisible(Lib: TLibrary);
var a, b: Integer;
begin
 for a:= 0 to High(Lib) do
  If Lib[a].Map[0].Index = TransparentBrick then Break;
 If a <= High(Lib) then begin
  Lba2Invisible.Idx.Lt:= a;
  Lba2Invisible.Idx.Brk:= 0;
  Lba2Invisible.BrkNr:= TransparentBrick;
  Lba2Invisible.Frame:= [foSelect];
  Lba2Invisible.Shape:= 1;
 end
 else begin
  for a:= 0 to High(Lib) do begin
   for b:= 0 to High(Lib[a].Map) do
    If Lib[a].Map[b].Index = TransparentBrick then Break;
   If (b <= High(Lib[a].Map)) and (Lib[a].Map[b].Index = TransparentBrick) then Break;
  end;
  If (a <= High(Lib)) and (b <= High(Lib[a].Map)) then begin
   Lba2Invisible.Idx.Lt:= a;
   Lba2Invisible.Idx.Brk:= b;
   Lba2Invisible.BrkNr:= TransparentBrick;
   Lba2Invisible.Frame:= [foSelect];
   Lba2Invisible.Shape:= 1;
  end
  else lba2Invisible.Idx.Lt:= 0;
 end;
end;

procedure GridToMap(Grd: TGrid; Lib: TLibrary); //full grids only
var a, b, c, Height, BlockCount: Integer;
    MapItem: TMapItem;
begin
 SetMapsLengths(63,24,63);
 ClearMap;
 BadLibraryMessage:= False;
 for a:= 0 to High(Grd.Cells) do begin
  Height:= 0;
  for b:= 0 to High(Grd.Cells[a].Objects) do begin
   BlockCount:= (Grd.Cells[a].Objects[b].Flags and $1F)+1;
   case (Grd.Cells[a].Objects[b].Flags and $E0) of
    $40: for c:= 0 to BlockCount - 1 do begin
          MapItem.Idx:= Grd.Cells[a].Objects[b].Blocks[c];
          SetBrkNr(Lib,MapItem);
          Map[a mod 64,Height,a div 64]:= MapItem;
          Inc(Height);
         end;
    $80: begin
          MapItem.Idx:= Grd.Cells[a].Objects[b].Blocks[0];
          SetBrkNr(Lib,MapItem);
          for c:= 0 to BlockCount - 1 do begin
           Map[a mod 64,Height,a div 64]:= MapItem;
           Inc(Height);
          end;
         end;
    $00: for c:= 0 to BlockCount - 1 do begin
          MapItem.Idx.Lt:= 0;
          MapItem.Idx.Brk:= 0;
          MapItem.BrkNr:= -1;
          Map[a mod 64,Height,a div 64]:= MapItem;
          Inc(Height);
         end;
   end;
  end;
 end;
 
 for c:=0 to 63 do
  for b:=0 to 24 do
   for a:=0 to 63 do
    SetFrameOpts(Map[a,b,c]);

 SetAllFrameLines;
end;

procedure FragmentToMap(Fr: TFragment; Lib: TLibrary);
var a, b, c: Integer;
    MapItem: TMapItem;
begin
 SetMapsLengths(Fr.X-1, Fr.Y-1, Fr.Z-1);
 ClearMap;
 BadLibraryMessage:= False;
 for c:= 0 to Fr.Z - 1 do
  for b:= 0 to Fr.Y - 1 do
   for a:= 0 to Fr.X - 1 do begin
    MapItem.Idx:= Fr.Blocks[a,b,c];
    SetBrkNr(Lib,MapItem);
    Map[a,b,c]:= MapItem;
    SetFrameOpts(Map[a,b,c]);
   end;
end;

Procedure DataToMap;
begin
 If GridNow then GridToMap(Grid,VLibrary)
 else FragmentToMap(Frag,VLibrary);
 FindLba2Invisible(VLibrary);
end;

Function GetOType(x, y, z: Integer): Byte;  //full grids only
begin
 If (Map[x,y,z].BrkNr = -1) then Result:=$00
 else if (y < 24)
  and (Map[x,y,z].Idx.Lt  = Map[x,y+1,z].Idx.Lt)
  and (Map[x,y,z].Idx.Brk = Map[x,y+1,z].Idx.Brk) then Result:= $80
 else if (y > 0)
  and (Map[x,y,z].Idx.Lt  = Map[x,y-1,z].Idx.Lt)
  and (Map[x,y,z].Idx.Brk = Map[x,y-1,z].Idx.Brk) then Result:= $80
 else Result:= $40;
end;

Function OTypeUpdate(x, y, z: Integer): Boolean;
begin
 Result:= (y > 0)
      and ((Map[x,y,z].Idx.Lt  <> Map[x,y-1,z].Idx.Lt)
        or (Map[x,y,z].Idx.Brk <> Map[x,y-1,z].Idx.Brk));
end;

Function FindSameCell(Nr: Integer): Integer;  //full grids only
var a, b: Integer;
begin
 Result:=-1;
 for a:=0 to Nr-1 do begin
  for b:=0 to 24 do
   If (Map[a mod 64,b,a div 64].Idx.Lt<>Map[Nr mod 64,b,Nr div 64].Idx.Lt)
   or (Map[a mod 64,b,a div 64].Idx.Brk<>Map[Nr mod 64,b,Nr div 64].Idx.Brk) then Break;
  If b<25 then Continue;
  Result:=a;
  Exit;
 end;
end;

Procedure MakeLibUsage(var Target: TGrid);   //full grids only
var a, b, c: Integer;
    Lt: Byte;
begin
 for a:=0 to 31 do
  Target.LibUsage[a]:=0;
 {If SetForm.cbSaveSuperCompat.Checked then
  for a:=1 to High(LLibrary) do
   Target.LibUsage[a div 8]:= Target.LibUsage[a div 8] or ($80 shr (a mod 8))
 else}
  for a:=0 to 63 do
   for b:=0 to 24 do
    for c:=0 to 63 do begin
     Lt:= Map[a,b,c].Idx.Lt;
     If Lt>0 then
      Target.LibUsage[Lt div 8]:= Target.LibUsage[Lt div 8] or ($80 shr (Lt mod 8));
    end;
end;

Function MapToGrid: TGrid;      //full grids only
var a, b, c, d, Start, Off: Integer;
    OType, NextOType: Byte; //$00-empty, $40-grouping, $80-repeating
begin
 MakeLibUsage(Result);
 Result.Lba2:= Grid.Lba2;
 Result.LibIndex:= Grid.LibIndex;
 Result.FragIndex:= Grid.FragIndex;
 With Result do begin
  SetLength(Cells,64*64);
  Off:= 8192;
  for c:= 0 to 63 do
   for a:= 0 to 63 do
    With Cells[a+c*64] do begin
     d:= FindSameCell(a+c*64);
     Repeated:= d > -1;
     If Repeated then begin
      Offset:= Cells[d].Offset;
      Continue;
     end;
     Offset:= Off;
     ObjCount:= 0;
     SetLength(Objects,0);
     Start:= 0;
     OType:= GetOType(a,0,c);
     for b:= 1 to 25 do begin
      If b < 25 then NextOType:= GetOType(a,b,c);
      If (NextOType <> OType) or (b = 25) or OTypeUpdate(a,b,c) then begin
       Inc(ObjCount);
       SetLength(Objects,ObjCount);
       case OType of
        $00: begin
              Objects[ObjCount-1].Flags:= b - Start - 1;
              SetLength(Objects[ObjCount-1].Blocks,0);
              Inc(Off,1);
             end;
        $40: begin
              Objects[ObjCount-1].Flags:= $40 or (b - Start - 1);
              SetLength(Objects[ObjCount-1].Blocks,b-Start);
              for d:= Start to b - 1 do Objects[ObjCount-1].Blocks[d-Start]:= Map[a,d,c].Idx;
              Inc(Off, 1 + (b-Start) * 2);
             end;
        $80: begin
              Objects[ObjCount-1].Flags:= $80 or (b - Start - 1);
              SetLength(Objects[ObjCount-1].Blocks,1);
              Objects[ObjCount-1].Blocks[0]:= Map[a,Start,c].Idx;
              Inc(Off,3);
             end;
       end;
       Start:= b;
       OType:= NextOType;
      end;
     end;
     Inc(Off,1);
    end;
 end;
end;

Function MapToFragment: TFragment;
var a, b, c: Integer;
    BlockIdx: TBlockIndex;
begin
 Result.X:= Length(Map);
 Result.Y:= Length(Map[0]);
 Result.Z:= Length(Map[0,0]);
 SetLength(Result.Blocks,Result.X,Result.Y,Result.Z);
 for c:= 0 to Result.Z - 1 do
  for b:= 0 to Result.Y - 1 do
   for a:= 0 to Result.X - 1 do
    Result.Blocks[a,b,c]:= Map[a,b,c].Idx;
end;

Function LayoutToPiece(LtNr: Integer): TPiece;
var a, b, c, x: Integer;
    Lt: TLayout;
begin
 If High(VLibrary) < LtNr then Exit;
 Lt:= VLibrary[LtNr];
 Result.X:= ClipBox.x2 - ClipBox.x1 + 1;
 Result.Y:= ClipBox.y2 - ClipBox.y1 + 1;
 Result.Z:= ClipBox.z2 - ClipBox.z1 + 1;
 SetLength(Result.Map,Result.X,Result.Y,Result.Z);
 for c:=0 to Result.Z - 1 do
  for b:=0 to Result.Y - 1 do
   for a:=0 to Result.X - 1 do begin
    x:= ((c + ClipBox.z1)*Lt.Y + b + ClipBox.y1)*Lt.X + a + ClipBox.x1;
    Result.Map[a,b,c].Idx.Lt:= LtNr;
    Result.Map[a,b,c].Idx.Brk:= x;
    Result.Map[a,b,c].BrkNr:= Lt.Map[x].Index;
    Result.Map[a,b,c].Shape:= Lt.Map[x].Shape;
    SetFrameOpts(Result.Map[a,b,c]);
   end;
end;

Function CopyPiece(x, y, z, dx, dy, dz: Word): TPiece;
var a, b, c: Integer;
begin
 Result.X:= Min(dx,HighX+1-x);
 Result.Y:= Min(dy,HighY+1-y);
 Result.Z:= Min(dz,HighZ+1-z);
 SetLength(Result.Map,Result.X,Result.Y,Result.Z);
 for a:=0 to Result.X-1 do
  for b:=0 to Result.Y-1 do
   for c:=0 to Result.Z-1 do
    If not (Form1.btInv.Down and (foNormal in Map[a+x,b+y,c+z].Frame)) then
     Result.Map[a,b,c]:=Map[a+x,b+y,c+z]
    else Result.Map[a,b,c]:=EmptyItem;
end;

Procedure PutPiece(x, y, z: Word; Fr: TPiece; Transparent: Boolean = True);
var a, b, c, ma, mb, mc: Integer;
begin
 ma:= Min(Fr.X,HighX+1-x) - 1;
 mb:= Min(Fr.Y,HighY+1-y) - 1;
 mc:= Min(Fr.Z,HighZ+1-z) - 1;
 for a:=0 to ma do
  for b:=0 to mb do
   for c:=0 to mc do
    If not Transparent or (foSelect in Fr.Map[a,b,c].Frame) then
     Map[a+x,b+y,c+z]:= Fr.Map[a,b,c];
 for a:=0 to ma do
  for b:=0 to mb do
   for c:=0 to mc do
    SetFrameLines(a+x,b+y,c+z);
 UpdateThumbnail(x,y,z,x+Fr.X,y+Fr.Y,z+Fr.Z);
end;

Procedure DelPiece(x, y, z, dx, dy, dz: Integer); overload;
var a, b, c: Integer;
begin
 For a:=Max(x,0) to Min(x+dx,HighX) do
  for b:=Max(y,0) to Min(y+dy,HighY) do
   for c:=Max(z,0) to Min(z+dz,HighZ) do
    If not (Form1.btInv.Down and (foNormal in Map[a,b,c].Frame)) then
     Map[a,b,c]:= EmptyItem;
 UpdateThumbnail(x,y,z,x+dx,y+dy,z+dz);
end;

Procedure DelPiece(Range: TBox); overload;
var a, b, c: Integer;
begin
 SetUndo;
 For a:=Max(Range.x1,0) to Min(Range.x2,HighX) do
  for b:=Max(Range.y1,0) to Min(Range.y2,HighY) do
   for c:=Max(Range.z1,0) to Min(Range.z2,HighZ) do
    If not (Form1.btInv.Down and (foNormal in Map[a,b,c].Frame)) then
     Map[a,b,c]:=EmptyItem;
 UpdateThumbnail(Range.x1,Range.y1,Range.z1,Range.x2,Range.y2,Range.z2);
end;

end.
